WHENEVER SQLERROR EXIT FAILURE ROLLBACK;
WHENEVER OSERROR  EXIT FAILURE ROLLBACK;

spool sys_message_pkg.log

create or replace package sys_message_pkg authid definer as

  /*
  ** SET_NAME - sets the message name
  */
  procedure set_code(code in varchar2);
  pragma restrict_references(set_code,
                             wnds);

  /*
  ** SET_TOKEN - defines a message token with a value
  */
  procedure set_token(token     in varchar2,
                      value     in varchar2,
                      translate in boolean default false);
  pragma restrict_references(set_token,
                             wnds);

  /*
  ** RETRIEVE - gets the message and token data, clears message buffer
  */
  procedure retrieve(msgout out varchar2);
  pragma restrict_references(retrieve,
                             wnds);

  /*
  ** CLEAR - clears the message buffer
  */
  procedure clear;
  pragma restrict_references(clear,
                             wnds);

  /*
  **  GET_STRING- get a particular translated message
  **       from the message dictionary database.
  **
  **  This is a one-call interface for when you just want to get a
  **  message without doing any token substitution.
  **  Returns NULL if the message cannot be found.
  */
  function get_string(codein in varchar2) return varchar2;
  pragma restrict_references(get_string,
                             wnds,
                             wnps);

  /*
  **  GET- get a translated and token substituted message
  **       from the message dictionary database.
  **       Returns NULL if the message cannot be found.
  */
  function get return varchar2;
  pragma restrict_references(get,
                             wnds);

  /*
  ** raise_error - raises the error to the calling entity
  **               via raise_application_error() prodcedure
  */
  procedure raise_error;
  pragma restrict_references(raise_error,
                             wnds,
                             wnps);

  procedure delete_message(p_message_code varchar2);

  procedure insert_message(p_message_code    varchar2,
                           p_message_name    varchar2,
                           p_message_text    varchar2,
                           p_language_code   varchar2 default userenv('lang'),
                           p_module_id       number default null,
                           p_last_updated_by number default 1,
                           p_created_by      number default 1);

  procedure update_message(p_message_code    varchar2,
                           p_message_name    varchar2,
                           p_message_text    varchar2,
                           p_language_code   varchar2 default userenv('lang'),
                           p_module_id       number default null,
                           p_last_updated_by number default 1);

end sys_message_pkg;
/
create or replace package body sys_message_pkg as

  msgcode varchar2(200);
  msgdata varchar2(2000);
  msgset  boolean := false;

  procedure set_code(code in varchar2) is
  begin
    msgcode := code;
    msgdata := '';
    msgset  := true;
  end;

  procedure set_token(token     in varchar2,
                      value     in varchar2,
                      translate in boolean default false) is
    flag varchar2(1);
  begin
    if translate then
      flag := 'Y';
    else
      flag := 'N';
    end if;
    /* Note that we are intentionally using chr(0) rather than */
    /* FND_GLOBAL.LOCAL_CHR() for a performance bug (982909) */
    msgdata := msgdata || flag || chr(0) || token || chr(0) || value ||
               chr(0);
  end set_token;

  /* This procedure is only to be called by the ATG; */
  /*  not for external use */
  procedure retrieve(msgout out varchar2) is
    out_val varchar2(2000);
  begin
    if msgset then
      /* Note that we are intentionally using chr(0) rather than */
      /* FND_GLOBAL.LOCAL_CHR() for a performance bug (982909) */
      out_val := msgcode || chr(0) || msgdata;
      msgset  := false;
    else
      out_val := '';
    end if;
  
    msgout := out_val;
  end;

  procedure clear is
  begin
    msgset := false;
  end;

  procedure raise_error is
  begin
    /* Note that we are intentionally using chr(0) rather than */
    /* FND_GLOBAL.LOCAL_CHR() for a performance bug (982909) */
    raise_application_error(-20001,
                            msgcode || ': ' ||
                            replace(rtrim(msgdata,
                                          chr(0)),
                                    chr(0),
                                    ', '));
  end;

  /*
  **  GET_STRING- get a particular translated message
  **       from the message dictionary database.
  **
  **  This is a one-call interface for when you just want to get a
  **  message without doing any token substitution.
  **  Returns NAMEIN (Msg name)  if the message cannot be found.
  */
  function get_string(codein in varchar2) return varchar2 is
    msg varchar2(2000) := null;
    cursor c1(code_arg varchar2) is
      select message
        from sys_messages m
       where code_arg = m.message_code
         and m.language = userenv('LANG');
    cursor c2(code_arg varchar2) is
      select message
        from sys_messages m
       where code_arg = m.message_code
         and 'ZHS' = m.language;
  begin
    /* get the message text out of the table */
    open c1(upper(codein));
    fetch c1
      into msg;
    if (c1%notfound) then
      open c2(upper(codein));
      fetch c2
        into msg;
      if (c2%notfound) then
        msg := codein;
      end if;
      close c2;
    end if;
    close c1;
    /* double ampersands don't have anything to do with tokens, they */
    /* represent access keys.  So we translate them to single ampersands*/
    /* so that the access key code will recognize them. */
    msg := substrb(replace(msg,
                           '',
                           '#'),
                   1,
                   2000);
    return msg;
  end;

  /*
  **  GET- get a translated and token substituted message
  **       from the message dictionary database.
  **       Returns NULL if the message cannot be found.
  */
  function get return varchar2 is
    msg       varchar2(2000);
    tok_nam   varchar2(30);
    tok_val   varchar2(2000);
    srch      varchar2(2000);
    flag      varchar2(1);
    pos       number;
    nextpos   number;
    data_size number;
    tslate    boolean;
  begin
    if (not msgset) then
      msg := '';
      return msg;
    end if;
    msg := get_string(msgcode);
    if ((msg is null) or (msg = '')) then
      msg := msgcode;
    end if;
    pos       := 1;
    data_size := length(msgdata);
    while pos < data_size loop
      flag := substr(msgdata,
                     pos,
                     1);
      pos  := pos + 2;
      /* Note that we are intentionally using chr(0) rather than */
      /* FND_GLOBAL.LOCAL_CHR() for a performance bug (982909) */
      nextpos := instr(msgdata,
                       chr(0),
                       pos);
      tok_nam := substr(msgdata,
                        pos,
                        nextpos - pos);
      pos     := nextpos + 1;
      nextpos := instr(msgdata,
                       chr(0),
                       pos);
      tok_val := substr(msgdata,
                        pos,
                        nextpos - pos);
      pos     := nextpos + 1;
    
      if (flag = 'Y') then
        /* translated token */
        tok_val := get_string(tok_val);
      end if;
      srch := '#' || tok_nam;
      if (instr(msg,
                srch) <> 0) then
        msg := substrb(replace(msg,
                               srch,
                               tok_val),
                       1,
                       2000);
      else
        /* try the uppercased version of the token name in case */
        /* the caller is (wrongly) passing a mixed case token name */
        /* Because now (July 99) all tokens in msg text should be */
        /* uppercase. */
        srch := '#' || upper(tok_nam);
        if (instr(msg,
                  srch) <> 0) then
          msg := substrb(replace(msg,
                                 srch,
                                 tok_val),
                         1,
                         2000);
        else
          msg := substrb(msg || ' (' || tok_nam || '=' || tok_val || ')',
                         1,
                         2000);
        end if;
      end if;
    end loop;
    /* double ampersands don't have anything to do with tokens, they */
    /* represent access keys.  So we translate them to single ampersands*/
    /* so that the access key code will recognize them. */
    msg    := substrb(replace(msg,
                              '',
                              '#'),
                      1,
                      2000);
    msgset := false;
    return msg;
  end;

  procedure delete_message(p_message_code varchar2) as
  begin
    delete sys_messages where message_code = upper(p_message_code);
  end delete_message;

  procedure insert_message(p_message_code    varchar2,
                           p_message_name    varchar2,
                           p_message_text    varchar2,
                           p_language_code   varchar2 default userenv('lang'),
                           p_module_id       number default null,
                           p_last_updated_by number default 1,
                           p_created_by      number default 1) as
  
  begin
  
    insert into sys_messages
      (message_code,
       language,
       message_name,
       message,
       module_id,
       creation_date,
       created_by,
       last_update_date,
       last_updated_by)
    values
      (upper(p_message_code),
       upper(p_language_code),
       p_message_name,
       p_message_text,
       p_module_id,
       sysdate,
       p_created_by,
       sysdate,
       p_last_updated_by);
  exception
    when dup_val_on_index then
      --insert  Ψһ
      sys_raise_app_error_pkg.raise_user_define_error(p_message_code            => 'SYS_MESSAGE_UNIQUE_ERROR',
                                                      p_created_by              => p_created_by,
                                                      p_package_name            => 'sys_message_pkg',
                                                      p_procedure_function_name => 'insert_message');
      raise_application_error(sys_raise_app_error_pkg.c_error_number,
                              sys_raise_app_error_pkg.g_err_line_id);
    when others then
      sys_raise_app_error_pkg.raise_sys_others_error(p_message                 => dbms_utility.format_error_backtrace || ' ' ||
                                                                                  sqlerrm,
                                                     p_created_by              => p_last_updated_by,
                                                     p_package_name            => 'sys_message_pkg',
                                                     p_procedure_function_name => 'insert_message');
    
      raise_application_error(sys_raise_app_error_pkg.c_error_number,
                              sys_raise_app_error_pkg.g_err_line_id);
  end insert_message;

  procedure update_message(p_message_code  varchar2,
                           p_message_name  varchar2,
                           p_message_text  varchar2,
                           p_language_code varchar2 default userenv('lang'),
                           p_module_id     number default null,
                           
                           p_last_updated_by number default 1) is
  begin
    update sys_messages
       set message_name     = p_message_name,
           message          = p_message_text,
           last_update_date = sysdate,
           last_updated_by  = p_last_updated_by,
           module_id        = p_module_id
     where message_code = upper(p_message_code)
       and language = upper(p_language_code);
  
  exception
    when others then
      sys_raise_app_error_pkg.raise_sys_others_error(p_message                 => dbms_utility.format_error_backtrace || ' ' ||
                                                                                  sqlerrm,
                                                     p_created_by              => p_last_updated_by,
                                                     p_package_name            => 'sys_message_pkg',
                                                     p_procedure_function_name => 'update_message');
    
      raise_application_error(sys_raise_app_error_pkg.c_error_number,
                              sys_raise_app_error_pkg.g_err_line_id);
  end update_message;

end sys_message_pkg;
/
spool off

exit
